<?php

namespace App\Http\Controllers;

use App\TakeProfit;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;
use App\Utils\ModuleUtil;
use App\Utils\Util;
use Illuminate\Support\Facades\Log;

class TakeProfitController extends Controller
{

    /**
     * Constructor
     *
     * @param  Util  $commonUtil
     * @param  ModuleUtil  $moduleUtil
     * @return void
     */
    public function __construct(public Util $commonUtil, public ModuleUtil $moduleUtil){}

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        if (! auth()->user()->can('take_profit.view') && ! auth()->user()->can('take_profit.create')) {
            abort(403, 'Unauthorized action.');
        }

        if (request()->ajax()) {
            return Datatables::of(TakeProfit::query())
                ->addColumn(
                    'action',
                    '<button data-href="{{action(\'App\Http\Controllers\TakeProfitController@edit\', [$id])}}" class="btn btn-xs btn-primary edit_take_profit_button"><i class="glyphicon glyphicon-edit"></i> @lang("messages.edit")</button>
                        &nbsp;
                        @if(empty($total_pv))
                        <button data-href="{{action(\'App\Http\Controllers\TakeProfitController@destroy\', [$id])}}" class="btn btn-xs btn-danger delete_take_profit_button"><i class="glyphicon glyphicon-trash"></i> @lang("messages.delete")</button>
                        @endif'
                )
                ->editColumn('amount', function($data) {
                    return $this->commonUtil->num_f($data->amount, true);
                })
                ->editColumn('created_at', function($data) {
                    return $data->created_at->format('d-m-Y g:i A');
                })
                ->removeColumn('id')
                ->removeColumn('updated_at')
                ->rawColumns([3])
                ->make(false);
        }



        return view('take_profit.index', [
            'total_take_profit' => $this->commonUtil->num_f(TakeProfit::sum('amount'), true)
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        if (! auth()->user()->can('take_profit.create')) {
            abort(403, 'Unauthorized action.');
        }
        return view('take_profit.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        if (! auth()->user()->can('take_profit.create')) {
            abort(403, 'Unauthorized action.');
        }
        try {
            $profit = TakeProfit::create($request->only(['note','amount']));

            $output = [
                'success' => true,
                'data' => $profit,
                'msg' => 'Take profit added succesfully',
            ];
        } catch (\Exception $e) {
            Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            $output = [
                'success' => false,
                'msg' => 'Something went wrong, please try again',
            ];
        }

        return $output;
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\TakeProfit  $takeProfit
     * @return \Illuminate\Http\Response
     */
    public function edit(TakeProfit $takeProfit)
    {
        if (! auth()->user()->can('take_profit.update')) {
            abort(403, 'Unauthorized action.');
        }
        if (request()->ajax()) {
            return view('take_profit.edit', compact('takeProfit'));
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\TakeProfit  $takeProfit
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, TakeProfit $takeProfit)
    {
        if (! auth()->user()->can('take_profit.update')) {
            abort(403, 'Unauthorized action.');
        }
        if (request()->ajax()) {
            try {
                $takeProfit->update($request->only(['note','amount']));
                $output = [
                    'success' => true,
                    'msg' => 'Take profit updated succesfully',
                ];
            } catch (\Exception $e) {
                Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

                $output = ['success' => false,
                    'msg' => 'Something went wrong, please try again',
                ];
            }
            return $output;
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\TakeProfit  $takeProfit
     * @return \Illuminate\Http\Response
     */
    public function destroy(TakeProfit $takeProfit)
    {
        if (! auth()->user()->can('take_profit.delete')) {
            abort(403, 'Unauthorized action.');
        }
        if (request()->ajax()) {
            try {
                $takeProfit->delete();
                $output = [
                    'success' => true,
                    'msg' => 'Take profit deleted succesfully',
                ];
            } catch (\Exception $e) {
                Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

                $output = [
                    'success' => false,
                    'msg' => 'Something went wrong, please try again',
                ];
            }

            return $output;
        }
    }
}
