<?php

namespace App\Http\Controllers;

use App\Product;
use App\ReturnableProductStock;
use App\Utils\TransactionUtil;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Yajra\DataTables\Facades\DataTables;

class ReturnableProductStockController extends Controller
{

    public function __construct(public TransactionUtil $transactionUtil){}   

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        if (request()->ajax()) {
            return DataTables::of(ReturnableProductStock::query())
                ->addColumn(
                    'action',
                    '   <button data-href="{{action(\'App\Http\Controllers\ReturnableProductStockController@edit\', [$id])}}" class="btn btn-xs btn-primary edit_product_stock_button"><i class="glyphicon glyphicon-edit"></i> @lang("messages.edit")</button>
                        &nbsp;
                        <button data-href="{{action(\'App\Http\Controllers\ReturnableProductStockController@destroy\', [$id])}}" class="btn btn-xs btn-danger delete_product_stock_button"><i class="glyphicon glyphicon-trash"></i> @lang("messages.delete")</button>'
                )
                ->editColumn('product_id', function($data) {
                    return $data->product?->name;
                })
                ->addColumn('brand_id', function($data) {
                    return $data->product?->brand?->name ?? '';
                })
                ->editColumn('created_at', function($data) {
                    return $data->created_at->format('d-m-Y g:i A');
                })
                ->removeColumn('id')
                ->removeColumn('updated_at')
                ->rawColumns([4])
                ->make(false);
        }
        return view('product-stock.index', [
            'total_stock'      => ReturnableProductStock::sum('quantity'),
            'total_returnable' => $this->transactionUtil->getContactReturnableDue()
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $products = Product::forDropdown(1, true);
        return view('product-stock.create', compact('products'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'product_id' => ['required','integer','exists:products,id'],
            'quantity'   => ['required','integer'],
        ]);

        try {
            $stock = ReturnableProductStock::query()->where('product_id', $request->product_id)->first();
            if($stock) {
                $stock->update(['quantity' => $request->quantity]);
            }
            else {
                $stock = ReturnableProductStock::query()->create($validated);
            }

            $output = [
                'success' => true,
                'data' => $stock,
                'msg' => "Product stock added succesfully",
            ];
        } catch (\Exception $e) {
            Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            $output = [
                'success' => false,
                'msg' => 'Something went wrong, please try again',
            ];
        }
        return $output;
    }

    /**
     * Display the specified resource.
     */
    public function show(ReturnableProductStock $returnableProductStock)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(ReturnableProductStock $returnableProductStock)
    {
        $products = Product::forDropdown(1, true);
        return view('product-stock.edit', compact('products','returnableProductStock'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, ReturnableProductStock $returnableProductStock)
    {
        $validated = $request->validate([
            'product_id' => ['required','integer','exists:products,id'],
            'quantity'   => ['required','integer'],
        ]);

        try {
            $returnableProductStock->update($validated);

            $output = [
                'success' => true,
                'data' => $returnableProductStock,
                'msg' => "Product stock added succesfully",
            ];
        } catch (\Exception $e) {
            Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            $output = [
                'success' => false,
                'msg' => 'Something went wrong, please try again',
            ];
        }
        return $output;
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(ReturnableProductStock $returnableProductStock)
    {
        if (request()->ajax()) {
            try {
                $returnableProductStock->delete();
                $output = [
                    'success' => true,
                    'msg' => 'Product stock deleted succesfully',
                ];
            } catch (\Exception $e) {
                Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

                $output = [
                    'success' => false,
                    'msg' => 'Something went wrong, please try again',
                ];
            }

            return $output;
        }
    }
}
