<?php

namespace App\Http\Controllers;

use App\Contact;
use App\Product;
use App\ProductDue;
use App\ReturnableProductStock;
use App\Utils\TransactionUtil;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Yajra\DataTables\Facades\DataTables;

class ProductDueController extends Controller
{
    public function __construct(public TransactionUtil $transactionUtil){}   

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        // if (! auth()->user()->can('trade.view') && ! auth()->user()->can('trade.create')) {
        //    abort(403, 'Unauthorized action.');
        // }
        if (request()->ajax()) {
            $query = ProductDue::query()
            ->leftJoin('contacts', 'product_dues.contact_id', '=', 'contacts.id')
            ->leftJoin('products', 'product_dues.product_id', '=', 'products.id');

            $dues = $query->select(
                'product_dues.id',
                'contacts.name as contact',
                'products.name as product',
                'product_dues.quantity',
                'product_dues.type',
                'product_dues.created_at',
            );

            if(!empty(request()->get('contact_id'))) {
                $query->where('product_dues.contact_id', request()->get('contact_id'));
            }
            if(!empty(request()->get('type'))) {
                $query->where('product_dues.type', request()->get('type'));
            }

            return DataTables::of($dues)
                ->addColumn(
                    'action',
                    '   <button data-href="{{action(\'App\Http\Controllers\ProductDueController@edit\', [$id])}}" class="btn btn-xs btn-primary edit_product_due_button"><i class="glyphicon glyphicon-edit"></i> @lang("messages.edit")</button>
                        &nbsp;
                        <button data-href="{{action(\'App\Http\Controllers\ProductDueController@destroy\', [$id])}}" class="btn btn-xs btn-danger delete_product_due_button"><i class="glyphicon glyphicon-trash"></i> @lang("messages.delete")</button>'
                )
                ->editColumn('created_at', function($data) {
                    return $data->created_at->format('d-m-Y g:i A');
                })
                ->removeColumn('id')
                ->removeColumn('contact_id')
                ->removeColumn('product_id')
                ->rawColumns([5])
                ->make(false);
        }

        $business_id = request()->session()->get('user.business_id');

        $customers = Contact::contactDropdown($business_id);
        $types = ProductDue::forDropdown();

        return view('product-due.index', [
            'due'       => ProductDue::query()->where('type', ProductDue::DUE)->sum('quantity'),
            'collect'   => ProductDue::query()->where('type', ProductDue::COLLECTION)->sum('quantity'),
            'customers' => $customers,
            'types'     => $types,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $contacts = Contact::contactDropdown(1);
        $products = Product::forDropdown(1, true);
        $types    = ProductDue::forDropdown();
        return view('product-due.create', compact('contacts','products','types'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'contact_id' => ['required','integer','exists:contacts,id'],
            'product_id' => ['required','integer','exists:products,id'],
            'type'       => ['required','string'],
            'quantity'   => ['required','integer'],
        ]);

        try {
            $due = ProductDue::query()->create($validated);

            $stock = ReturnableProductStock::query()->where('product_id', $request->product_id)->first();
            if($stock) {
                if($request->type == ProductDue::COLLECTION) {
                    $stock->increment('quantity', $request->quantity);
                }
                else {
                    $stock->decrement('quantity', $request->quantity);
                }
            }

            $output = [
                'success' => true,
                'data' => $due,
                'msg' => "Product $request->type added succesfully",
            ];
        } catch (\Exception $e) {
            Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            $output = [
                'success' => false,
                'msg' => 'Something went wrong, please try again',
            ];
        }
        return $output;
    }

    /**
     * Display the specified resource.
     */
    public function show(ProductDue $productDue)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(ProductDue $productDue)
    {
        $contacts = Contact::contactDropdown(1);
        $products = Product::forDropdown(1, true);
        $types    = ProductDue::forDropdown();
        return view('product-due.edit', compact('contacts','products','types','productDue'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, ProductDue $productDue)
    {
        $validated = $request->validate([
            'contact_id' => ['nullable','integer','exists:contacts,id'],
            'product_id' => ['nullable','integer','exists:products,id'],
            'type'       => ['required','string'],
            'quantity'   => ['required','integer'],
        ]);

        try {
            $stock = ReturnableProductStock::query()->where('product_id', $productDue->product_id)->first();
            if($stock) {
                if($productDue->type == ProductDue::COLLECTION) {
                    $stock->decrement('quantity', $productDue->quantity);
                }
                else {
                    $stock->increment('quantity', $productDue->quantity);
                }
            }

            $productDue->update($validated);

            $stock = ReturnableProductStock::query()->where('product_id', $request->product_id)->first();
            if($stock) {
                if($request->type == ProductDue::COLLECTION) {
                    $stock->increment('quantity', $request->quantity);
                }
                else {
                    $stock->decrement('quantity', $request->quantity);
                }
            }
            $output = [
                'success' => true,
                'data' => $productDue,
                'msg' => "Product $request->type updated succesfully",
            ];
        } catch (\Exception $e) {
            Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            $output = [
                'success' => false,
                'msg' => 'Something went wrong, please try again',
            ];
        }
        return $output;
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(ProductDue $productDue)
    {
        if (request()->ajax()) {
            try {
                $stock = ReturnableProductStock::query()->where('product_id', $productDue->product_id)->first();
                if($stock) {
                    if($productDue->type == ProductDue::COLLECTION) {
                        $stock->decrement('quantity', $productDue->quantity);
                    }
                    else {
                        $stock->increment('quantity', $productDue->quantity);
                    }
                }
                
                $productDue->delete();
                $output = [
                    'success' => true,
                    'msg' => 'Product due deleted succesfully',
                ];
            } catch (\Exception $e) {
                Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

                $output = [
                    'success' => false,
                    'msg' => 'Something went wrong, please try again',
                ];
            }

            return $output;
        }
    }
}
