<?php

namespace App\Http\Controllers;

use App\CustomerGroup;
use App\Order;
use App\OrderItem;
use App\Product;
use App\Utils\BusinessUtil;
use App\Utils\ModuleUtil;
use App\Utils\ProductUtil;
use App\Utils\TransactionUtil;
use App\Variation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Yajra\DataTables\Facades\DataTables;

class OrderController extends Controller
{
    public $dummyPaymentLine;

    /**
     * Constructor
     * 
     * @return void
     */
    public function __construct(public ProductUtil $productUtil, public TransactionUtil $transactionUtil, public BusinessUtil $businessUtil, public ModuleUtil $moduleUtil)
    {

        $this->dummyPaymentLine = ['method' => 'cash', 'amount' => 0, 'note' => '', 'card_transaction_number' => '', 'card_number' => '', 'card_type' => '', 'card_holder_name' => '', 'card_month' => '', 'card_year' => '', 'card_security' => '', 'cheque_number' => '', 'bank_account_number' => '',
            'is_return' => 0, 'transaction_no' => '', ];
    }
    
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        if (! auth()->user()->can('order.view') && ! auth()->user()->can('order.create')) {
            abort(403, 'Unauthorized action.');
        }

        if (request()->ajax()) {
            return DataTables::of(Order::query())
                ->addColumn(
                    'action',
                    '   <a href="{{action(\'App\Http\Controllers\OrderController@edit\', [$id])}}" class="btn btn-xs btn-primary"><i class="glyphicon glyphicon-edit"></i> @lang("messages.edit")</a>
                        &nbsp;
                        <a href="#" data-href="{{action(\'App\Http\Controllers\OrderController@show\', [$id])}}" class="btn btn-xs btn-info btn-modal" data-container=".view_modal"><i class="glyphicon glyphicon-eye-open"></i> @lang("messages.view")</a>
                        <button data-href="{{action(\'App\Http\Controllers\OrderController@destroy\', [$id])}}" class="btn btn-xs btn-danger delete_product_stock_button"><i class="glyphicon glyphicon-trash"></i> @lang("messages.delete")</button>'
                )
                ->filter(function($data) {
                    if(request()->has('contact_id')) {
                        $data->where('contact_id', request()->get('contact_id'));
                    }
                }, true)
                ->editColumn('created_at', function($data) {
                    return $data->created_at->format('d-m-Y g:i A');
                })
                ->editColumn('business_id', function($data) {
                    return $data->business?->name ?? '';
                })
                ->editColumn('contact_id', function($data) {
                    return $data->contact?->name ?? '';
                })
                ->rawColumns(['action'])
                ->toJson();
        }
        return view('order.index');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        if (! auth()->user()->can('order.create')) {
            abort(403, 'Unauthorized action.');
        }

        $products = Product::forDropdown(1, true);

        $business_id = request()->session()->get('user.business_id');

        $types = [];
        if (auth()->user()->can('supplier.create')) {
            $types['supplier'] = __('report.supplier');
        }
        if (auth()->user()->can('customer.create')) {
            $types['customer'] = __('report.customer');
        }
        if (auth()->user()->can('supplier.create') && auth()->user()->can('customer.create')) {
            $types['both'] = __('lang_v1.both_supplier_customer');
        }
        $customer_groups = CustomerGroup::forDropdown($business_id);

        return view('order.create', compact('products', 'types', 'customer_groups'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        if (! auth()->user()->can('order.create')) {
            abort(403, 'Unauthorized action.');
        }

        if (! auth()->user()->can('purchase.create')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = $request->session()->get('user.business_id');

            $order = Order::query()->create([
                'business_id'      => $business_id,
                'contact_id'       => $request->contact_id,
                'additional_notes' => $request->additional_notes,
                'created_by' => auth()->id(),
            ]);

            foreach ($request->orders as $data) {
                $order->items()->create([
                    'product_id'   => $data['product_id'],
                    'variation_id' => $data['variation_id'],
                    'quantity'     => $data['quantity']
                ]);
            }

            //Generate reference number
            $order->ref_no = sprintf('%s%05s', '', $order->id);
            $order->save();

            $output = ['success' => 1,
                'msg' => __('lang_v1.order_add_success'),
            ];
        } catch (\Exception $e) {
            Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            $output = ['success' => 0,
                'msg' => __('messages.something_went_wrong'),
            ];
        }

        return redirect('order')->with('status', $output);
    }

    /**
     * Display the specified resource.
     */
    public function show(Order $order)
    {
        $order->load('items','items.product');

        if (! auth()->user()->can('order.view')) {
            abort(403, 'Unauthorized action.');
        }

        return view('order.show', compact('order'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Order $order)
    {
        if (! auth()->user()->can('order.update')) {
            abort(403, 'Unauthorized action.');
        }

        $order->load('items','items.product');

        $products = Product::forDropdown(1, true);

        $business_id = request()->session()->get('user.business_id');

        $types = [];
        if (auth()->user()->can('supplier.create')) {
            $types['supplier'] = __('report.supplier');
        }
        if (auth()->user()->can('customer.create')) {
            $types['customer'] = __('report.customer');
        }
        if (auth()->user()->can('supplier.create') && auth()->user()->can('customer.create')) {
            $types['both'] = __('lang_v1.both_supplier_customer');
        }
        $customer_groups = CustomerGroup::forDropdown($business_id);

        return view('order.edit', compact('products', 'types', 'customer_groups', 'order'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Order $order)
    {
        if (! auth()->user()->can('order.update')) {
            abort(403, 'Unauthorized action.');
        }

        if (! auth()->user()->can('purchase.update')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = $request->session()->get('user.business_id');

            $order->update([
                'business_id'      => $business_id,
                'contact_id'       => $request->contact_id,
                'additional_notes' => $request->additional_notes,
                'created_by' => auth()->id(),
            ]);
            foreach ($request->orders as $data) {
                $input = [
                    'product_id'   => $data['product_id'],
                    'variation_id' => $data['variation_id'],
                    'quantity'     => $data['quantity']
                ];
                if(isset($data['item_id'])) {
                    $order->items()->where('id', $data['item_id'])->update($input);
                }
                else {
                    $order->items()->create($input);
                }
            }

            $output = ['success' => 1,
                'msg' => __('lang_v1.order_update_success'),
            ];
        } catch (\Exception $e) {
            Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            $output = ['success' => 0,
                'msg' => __('messages.something_went_wrong'),
            ];
        }

        return redirect('order')->with('status', $output);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Order $order)
    {
        if (! auth()->user()->can('order.delete')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $order->delete();
            $output = ['success' => true, 'msg' => __('lang_v1.order_delete_success')];

        } catch (\Exception $e) {
            Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            $output = ['success' => false, 'msg' => $e->getMessage()];
        }

        return $output;
    }

    public function entryRow(Request $request)
    {
        if ($request->ajax()) {
            $product_id = $request->input('product_id');
            $variation_id = $request->input('variation_id');
            $business_id = $request->session()->get('user.business_id');
            $location_id = $request->input('location_id');
            $is_purchase_order = $request->has('is_purchase_order');
            $supplier_id = $request->input('supplier_id');

            $hide_tax = 'hide';
            if ($request->session()->get('business.enable_inline_tax') == 1) {
                $hide_tax = '';
            }

            if (! empty($product_id)) {
                $row_count = $request->input('row_count');
                $product   = Product::where('id', $product_id)->with(['unit', 'second_unit'])->first();

                $sub_units = $this->productUtil->getSubUnits($business_id, $product->unit->id, false, $product_id);
                $query = Variation::where('product_id', $product_id)
                                ->with([
                                    'product_variation',
                                    'variation_location_details' => fn ($q) => $q->where('location_id', $location_id)
                                ]);
                if ($variation_id !== '0') {
                    $query->where('id', $variation_id);
                }
                $variations = $query->get();

                return view('order.partials.order_entry_row')
                    ->with(compact(
                        'product',
                        'variations',
                        'row_count',
                        'variation_id',
                        'hide_tax',
                        'sub_units',
                        'is_purchase_order',
                    ));
            }
        }
    }

    public function itemDelete($id) 
    {
        try {
            $item = OrderItem::query()->findOrFail($id);
            $item->delete();
            $output = ['success' => true, 'msg' => __('lang_v1.order_delete_success')];

        } catch (\Exception $e) {
            Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            $output = ['success' => false, 'msg' => $e->getMessage()];
        }

        return $output;
    }
}
